package applets.clickableImage;

import java.applet.Applet;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Event;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Panel;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Scrollbar;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Constructor;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.JButton;

import applets.util.URLEncoder;

class CICImageCanvas extends Panel{

	private static final long	serialVersionUID	= 1L;
	// ClickableImageCreatorCanvas
	private Color CORRECT_REGION_BORDER = new Color(0x00, 0xc0, 0x00);
	private Color CORRECT_REGION_SHADE = new Color(0x00, 0x00, 0x00);
	private Color INCORRECT_REGION_BORDER = new Color(0x00, 0x00, 0x00);
	private Color INCORRECT_REGION_SHADE = new Color(0x00, 0x00, 0x00);
	private boolean supportsAlpha = false;
	private Image image;
	private Image passImage;
	private int image_width;
	private int image_height;
	private int image_x;
	private int image_y;
	private Scrollbar hbar;
	private Scrollbar vbar;
	public Point dragPoint;
	private boolean selecting;
	private boolean imageLoaded;
	// Test the behavior of Scrollbar to see whether "maximum" is the maximum value that the
	// slider can be set to (what the 1.1 API *says* should happen), or whether it's
	// the length of the rail, so that the maximum attainable value is maximum-visible (what the
	// 1.2 API says should be done, and what a lot of vendors seem to have done in 1.1).
	private static boolean scrollbarIsBuggy = false;
	private static boolean scrollbarTested = false;
	private Vector<Vector<Point>> regions;
	private int correct;
	private int active;
	private ClickableImageCreator myApplet;

	CICImageCanvas(Vector<Vector<Point>> regions, int correct, ClickableImageCreator myApplet) {

		setBackground(Color.gray);
		this.setLayout(new BorderLayout());

		this.regions = regions;
		this.correct = correct - 1;
		this.myApplet = myApplet;

		selecting = false;
		imageLoaded = false;

		try {
			Constructor cstr = Color.class.getConstructor(new Class[] { Integer.TYPE, Integer.TYPE,
					Integer.TYPE, Integer.TYPE });
			CORRECT_REGION_SHADE = (Color) cstr.newInstance(new Integer[] { new Integer(0),
					new Integer(0xff), new Integer(0), new Integer(0x20) });
			INCORRECT_REGION_SHADE = (Color) cstr.newInstance(new Integer[] { new Integer(0),
					new Integer(0), new Integer(0), new Integer(0x20) });
			supportsAlpha = true;
		} catch (Exception e) {
			// Accept default (non-alpha) values
		}

	}

	void addRegion() {

		// Regions get added on at the end. Only add on a new region if we've already added points
		// in to the region that we added on last
		if (((Vector) regions.lastElement()).size() > 0) {

			regions.addElement(new Vector());
			active = regions.size() - 1;
			dragPoint = null;
			repaint();
		}
	}

	// returns the first endpoint of the line that point lies closest to
	private int closestLineIn(int x, int y, Vector v) {

		double dist = Double.POSITIVE_INFINITY, trydist;
		int closest = -1;

		// now loop through the rest
		for (int i = 0; i < v.size() - 1; i++) {
			Point p1 = (Point) v.elementAt(i);
			Point p2 = (Point) v.elementAt(i + 1);
			double a = p2.x - p1.x;
			double b = p2.y - p1.y;

			// llen is the length of the line
			double llen = distance(p1.x, p1.y, p2.x, p2.y);
			// pdist is perpindicular distance from point to extended line
			double pdist = Math.abs(a * (y - p1.y) - b * (x - p1.x)) / llen;
			// d1 and d2 are distances to line endpoints
			double d1 = distance(x, y, p1.x, p1.y);
			double d2 = distance(x, y, p2.x, p2.y);

			// if we are farther than the line length from the farther endpoint, then we are not
			// right along
			// the line, and we can use the distance to the nearer endpoint as a good approximation
			// otherwise, if we are along the line then we can use pdist
			if (Math.max(d1, d2) > llen)
				trydist = Math.min(d1, d2);
			else
				trydist = pdist;
			if (trydist < dist) {
				dist = trydist;
				closest = i;
			}
		}
		return closest;
	}

	void deleteActiveRegion() {

		if (active == correct) {

			correct = -1;
		} else if (correct > active) {

			--correct;
		}

		regions.removeElementAt(active);
		active = 0;

		if (regions.size() == 0) {

			regions.addElement(new Vector<Point>());
		}

		repaint();
	}

	private double distance(float x1, float y1, float x2, float y2) {

		return Math.sqrt((x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1));
	}

	private void drawActiveRegion(Graphics g) {

		Vector<Point> v = this.regions.elementAt(this.active);

		if (v == null || v.size() == 0)
			return;

		int i;

		g.setPaintMode();
		g.setColor(active == correct ? CORRECT_REGION_BORDER : INCORRECT_REGION_BORDER);

		Polygon p = regionToPolygon(v, -image_x, -image_y);

		for (i = 0; i < p.npoints - 1; i++) {

			drawThickLine(g, p.xpoints[i], p.ypoints[i], p.xpoints[i + 1], p.ypoints[i + 1]);
		}
		drawThickLine(g, p.xpoints[i], p.ypoints[i], p.xpoints[0], p.ypoints[0]);

		g.setColor(active == correct ? CORRECT_REGION_SHADE : INCORRECT_REGION_SHADE);
		if (!supportsAlpha) {
			g.setXORMode(new Color(0x80, 0x80, 0x80));
		}
		g.fillPolygon(p);

		g.setPaintMode();
		g.setColor(Color.red);
		for (i = 0; i < p.npoints; i++)
			g.fill3DRect(p.xpoints[i] - 1, p.ypoints[i] - 1, 4, 4, true);

		if (dragPoint != null) {
			g.setColor(Color.yellow);
			g.fill3DRect(dragPoint.x - image_x - 1, dragPoint.y - image_y - 1, 4, 4, true);
		}
	}

	// passImage is an image that shows the basic image with the inactive regions
	// marked out. The active region is drawn on top of this.
	private void drawPassImage() {

		if (!imageLoaded)
			return;

		if (passImage == null) {

			passImage = this.createImage(image_width, image_height);
		}
		if (passImage == null)
			return;

		Graphics g = passImage.getGraphics();
		g.drawImage(image, 0, 0, this);

		for (int i = 0; i < regions.size(); i++) {

			if (i != active)
				drawPassiveRegion(g, i);
		}
	}

	private void drawPassiveRegion(Graphics g, int index) {

		Vector v = (Vector) this.regions.elementAt(index);

		if (v == null || v.size() == 0)
			return;

		Polygon p = regionToPolygon(v, 0, 0);

		g.setColor(index == correct ? CORRECT_REGION_SHADE : INCORRECT_REGION_SHADE);
		if (!supportsAlpha) {
			g.setXORMode(new Color(0x7f, 0x7f, 0x7f));
		}
		g.fillPolygon(p);

		g.setPaintMode();
		g.setColor(index == correct ? CORRECT_REGION_BORDER : INCORRECT_REGION_BORDER);

		// would be g.drawPolygon here, but netscape screws that up
		int i;
		for (i = 0; i < p.npoints - 1; i++) {

			g.drawLine(p.xpoints[i], p.ypoints[i], p.xpoints[i + 1], p.ypoints[i + 1]);
		}
		g.drawLine(p.xpoints[i], p.ypoints[i], p.xpoints[0], p.ypoints[0]);

	}

	private void drawThickLine(Graphics g, int x1, int y1, int x2, int y2) {

		g.drawLine(x1, y1, x2, y2);
		g.drawLine(x1 + 1, y1, x2 + 1, y2);
		g.drawLine(x1, y1 + 1, x2, y2 + 1);
	}

	private Point findClosePointIn(int x, int y, Vector v) {

		// returns first close point found; may not be closest...
		for (int i = 0; i < v.size() - 1; i++) {
			Point p1 = (Point) v.elementAt(i);
			int trydist2 = (p1.x - x) * (p1.x - x) + (p1.y - y) * (p1.y - y);
			if (trydist2 < 9)
				return p1; // distance of 3 pixels is close enough...
		}
		return null;
	}

	private int findRegion(int x, int y) {

		// return region index of region containing x,y

		for (int i = 0; i < regions.size(); i++) {

			Vector v = (Vector) regions.elementAt(i);

			if (v.size() > 3 && regionToPolygon(v, 0, 0).inside(x, y))
				return i;
		}
		return -1;
	}

	int getCorrect() {

		return correct;
	}

	Vector getRegions() {

		return regions;
	}

	private void gotImage() {

		imageLoaded = true;
		image = myApplet.image;

		image_width = image.getWidth(this);
		image_height = image.getHeight(this);

		int width = this.size().width;
		int height = this.size().height;

		// Determine whether we'll be needing horizontal/vertical scrolls
		vbar = new Scrollbar(Scrollbar.VERTICAL);
		hbar = new Scrollbar(Scrollbar.HORIZONTAL);

		int view_width = image_height > height ? width - vbar.preferredSize().width : width;
		int view_height = image_width > width ? height - hbar.preferredSize().height : height;

		if (image_height > view_height) {

			if (hasBuggyScrollbarAPI()) {

				// This attempts to fix a BUG in Netscape 4.5, which does NOT follow the AWT spec
				// for setting scrollbars. Doh!
				vbar.setValues(0, view_height, 0, image_height);
				System.out.println("special: " + view_height + "  " + image_height);
			} else {

				vbar.setValues(0, view_height, 0, image_height - view_height);
				System.out.println("not special: " + view_height + "  "
						+ (image_height - view_height));

			}
			add("East", vbar);
			image_y = 0;
		} else {
			image_y = (image_height - height) / 2;
		}

		if (image_width > view_width) {

			if (hasBuggyScrollbarAPI()) {

				// This attempts to fix a BUG in Netscape 4.5, which does NOT follow the AWT spec
				// for setting scrollbars. Doh!
				hbar.setValues(0, view_width, 0, image_width);
			} else {
				hbar.setValues(0, view_width, 0, image_width - view_width);
			}
			add("South", hbar);
			image_x = 0;
		} else {
			image_x = (image_width - width) / 2;
		}

		this.validate();
	}

	public boolean handleEvent(Event e) {

		// handle scrolling of scroll bars
		if (e.target == hbar) {

			int ix = image_x;
			image_x = hbar.getValue();
			if (ix != image_x)
				update(getGraphics());
			return true;
		} else if (e.target == vbar) {

			int iy = image_y;
			image_y = vbar.getValue();
			if (iy != image_y)
				update(getGraphics());
			return true;
		}
		return super.handleEvent(e);
	}

	private static boolean hasBuggyScrollbarAPI() {

		if (!scrollbarTested) {

			Scrollbar testBar = new Scrollbar();
			testBar.setValues(0, 10, 0, 20);
			try {
				testBar.setValue(20);

				if (testBar.getValue() < 20)
					scrollbarIsBuggy = true;
			} catch (Exception e) {
				scrollbarIsBuggy = true;
			}

			scrollbarTested = true;
		}
		return scrollbarIsBuggy;
	}

	public boolean mouseDown(Event e, int x, int y) {

		if (selecting)
			return true;

		Point cp;
		int r;
		int ax = x + image_x;
		int ay = y + image_y;
		Vector v = (Vector) this.regions.elementAt(active);

		if (ax < 0 || ay < 0 || ax >= image_width || ay >= image_height)
			return false;

		if ((cp = findClosePointIn(ax, ay, v)) != null) {

			// check if they clicked on an existing point
			dragPoint = cp; // let them drag this point around
			dragPoint.move(ax, ay);

		} else if (((r = findRegion(ax, ay)) >= 0) && (r != active)) {

			// check if they clicked in an existing region; set them to select that region on
			// mouseup
			selecting = true;
		} else if (v.size() == 0) {

			dragPoint = new Point(ax, ay);

			// When starting a new region, the first and last points are identical
			v.addElement(dragPoint);
			v.addElement(dragPoint);
		} else {
			// break apart the nearest line
			dragPoint = new Point(ax, ay);

			int cl = closestLineIn(ax, ay, v);
			v.insertElementAt(dragPoint, closestLineIn(ax, ay, v) + 1);
		}
		repaint();
		return true;
	}

	public boolean mouseDrag(Event e, int x, int y) {

		if (selecting)
			return true;
		if (dragPoint == null)
			return false;
		int ax = x + image_x;
		int ay = y + image_y;
		ax = ax < 0 ? 0 : (ax >= image_width ? image_width - 1 : ax);
		ay = ay < 0 ? 0 : (ay >= image_height ? image_height - 1 : ay);
		dragPoint.move(ax, ay);
		repaint();
		return true;
	}

	public boolean mouseMove(Event e, int x, int y) {

		Point oldp = dragPoint;
		Vector v = (Vector) regions.elementAt(active);

		dragPoint = findClosePointIn(x + image_x, y + image_y, v);

		if (dragPoint != oldp) {

			Graphics g = this.getGraphics();
			g.clipRect(-image_x, -image_y, image_width, image_height);
			g.setPaintMode();
			if (oldp != null) {
				g.setColor(Color.red);
				g.fill3DRect(oldp.x - image_x - 1, oldp.y - image_y - 1, 4, 4, true);
			}
			if (dragPoint != null) {
				g.setColor(Color.yellow);
				g.fill3DRect(dragPoint.x - image_x - 1, dragPoint.y - image_y - 1, 4, 4, true);
			}
		}
		return true;
	}

	public boolean mouseUp(Event e, int x, int y) {

		if (selecting) {

			active = findRegion(x + image_x, y + image_y);
			selecting = false;
			repaint();
		}
		return true;
	}

	public void paint(Graphics g) {

		if (!imageLoaded) {

			g.clearRect(0, 0, size().width, size().height);

			if (myApplet.gotImage) {
				gotImage();
			} else {
				g.drawString(myApplet.loadingStatus, 4, 34);
			}
		}

		// not else; imageLoaded may have changed...
		if (imageLoaded) {

			drawPassImage();

			g.clipRect(-image_x, -image_y, image_width, image_height);

			if (passImage != null)
				g.drawImage(passImage, -image_x, -image_y, this);

			drawActiveRegion(g);
		}
	}

	private static Polygon regionToPolygon(Vector v, int xTrans, int yTrans) {

		int npoints = v.size() - 1;
		int xpoints[] = new int[npoints];
		int ypoints[] = new int[npoints];

		for (int i = 0; i < npoints; i++) {
			Point p1 = (Point) v.elementAt(i);
			xpoints[i] = p1.x + xTrans;
			ypoints[i] = p1.y + yTrans;
		}

		return new Polygon(xpoints, ypoints, npoints);
	}

	void selectActiveRegion() {

		correct = active;

		passImage = null; // Force a complete repaint of the canvas
		repaint();
	}

	public void update(Graphics g) {

		paint(g); // no need to clear for an update
	}
}

// Implemented in the java 1.02 API; things like Choice.remove() and scrollpanes from 1.1 would make
// it
// simpler and cleaner
public class ClickableImageCreator extends Applet implements Runnable,  ActionListener {

	/*
	 * NOTE: This MUST be the same as com.maplesoft.application.Constants.CLICKABLE_IMAGE_RESPONSE
	 * (but we can't cross-link them because they're on different class paths)
	 */
	final static String TEST_KEY = "clickable.image.response";
	Image image;
	private JButton undoButton;
	private JButton showAll;
	private JButton newButton;
	private JButton deleteButton;
	private JButton selectButton;
	private Panel topBar;
	private CICImageCanvas imageArea;
	String loadingStatus;
	boolean gotImage = false;
	// Something to attach dialogs to.
	private Frame holderFrame;

	/* The user's session locale. Initialize to en_US, update in init() method */
	Locale mtaLocale = new Locale("en", "US");	
	
	/**
	 * Handle clicks on the various button bar items.
	 */
	public void actionPerformed(ActionEvent event) {
		if (event.getSource() == newButton) {

			imageArea.addRegion();
		} else if (event.getSource() == deleteButton) {

			imageArea.deleteActiveRegion();
		} else if (event.getSource() == selectButton) {

			imageArea.selectActiveRegion();
		} 
		
		return;
	}
	
	// make an absolute URL from an URL relative to where the document was served from.
	private URL completeURL(String tail) throws MalformedURLException {

		try {
			return new URL(URLEncoder.encode(tail));
		} catch (MalformedURLException e) {
			// String urlBase = this.getDocumentBase().toString();
			String urlBase = this.getCodeBase().toString();

			int c = urlBase.indexOf('?');
			if (c >= 0)
				urlBase = urlBase.substring(0, c); // exclude ?
			c = urlBase.lastIndexOf('/');
			if (c >= 0)
				urlBase = urlBase.substring(0, c + 1); // include /
			return new URL(URLEncoder.encode(urlBase + tail));
		}
	}

	public void destroy() {

		this.holderFrame.dispose();
		this.holderFrame = null;
	}

	private void dialog(Exception e) {

		showStatus(loadingStatus = applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.LoadingStatus.ErrorLoadingImage"));
		System.out.println("Exception: " + e.toString());
		e.printStackTrace();

		popup(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.Popup.ErrorLoadingImage.Part1")+"\n\n" + "   "
				+ this.getParameter("imageURL") + "\n\n"
				+ applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.Popup.ErrorLoadingImage.Part2"));
	}

	// 
	// EVENT HANDLERS FOR THE BUTTONS
	//
	public void doneButtonAction() {

		Vector regions = imageArea.getRegions();
		int correct = imageArea.getCorrect() + 1;

		if (correct < 1 || correct > regions.size()) {

			popup(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.Popup.MustDesignateCorrectRegion"));
			return;
		}

		try {

			String urlStr = "dbEdit.AddClickableImage?actionID=accept"; // trusted to contain ?
			// include first the stuff that this applet has direct access to...
			urlStr += "&" + TEST_KEY + "=" + this.getParameter(TEST_KEY) + "&answer=" + correct
					+ "&imageURL=" + this.getParameter("imageURL") + "&width="
					+ image.getWidth(this) + "&height=" + image.getHeight(this);

			int n = 1;
			for (int i = 0; i < regions.size(); i++) {

				Vector v = (Vector) regions.elementAt(i);

				if (v.size() > 3) { // need at least 3 points for a real polygon

					urlStr += "&region." + n + "=";

					for (int j = 0; j < v.size() - 1; j++) {
						Point p1 = (Point) v.elementAt(j);
						urlStr += p1.x + "," + p1.y + ",";
					}
					n++;
				}
			}
			urlStr += "&numberOfRegions=" + (n - 1);
			urlStr += "&forceReload=" + System.currentTimeMillis(); // Make each URL unique so no
																	// caching

			if (n > 2) {

				this.getAppletContext().showDocument(completeURL(urlStr), "_self");
			} else {

				popup(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.Popup.NeedAtLeastTwoRegions"));
				return;
			}
		} catch (Exception e) {

			e.printStackTrace();

			showStatus(loadingStatus = applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.LoadingStatus.QuestionUpdateFailed"));
		}
	}

	private int getAnswerFromParameters() {

		try {
			return Integer.parseInt(this.getParameter("answer"));
		} catch (Exception e) {
			return -1;
		}
	}

	private Vector<Vector<Point>> getRegionsFromParameters() {

		// the elements of regions are vectors that contain the list of points for each region
		// those lists should have a redundant point, duplicated as the head and tail of the list
		Vector<Vector<Point>> regions = new Vector<Vector<Point>>();

		String region;
		for (int i = 1; (region = this.getParameter("region." + i)) != null; i++) {

			StringTokenizer stz = new StringTokenizer(region, ",\n\t ");
			int numPoints = stz.countTokens() / 2;

			Vector<Point> v = new Vector<Point>();

			for (int j = 0; j < numPoints; j++)
				v.addElement(new Point(Integer.parseInt(stz.nextToken()),
						Integer.parseInt(stz.nextToken())));

			v.addElement(v.elementAt(0)); // the first and last points are identical

			regions.addElement(v);
		}

		if (regions.size() == 0)
			regions.addElement(new Vector<Point>());

		return regions;
	}

	public void init() {

		String[] locale;
		
		try
		{
			locale = getParameter("mtalocale").split("_");
		}
		catch (Exception e)
		{
			locale = "en_US".split("_");
		}
		
		if (locale.length==1)
			mtaLocale = new Locale(locale[0]);
		else if (locale.length==2)
			mtaLocale = new Locale(locale[0], locale[1]);
		else if (locale.length==3)
			mtaLocale = new Locale(locale[0], locale[1], locale[2]);		
		
		this.holderFrame = new Frame();

		showStatus(loadingStatus = applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.LoadingStatus.LoadingImage"));

		// run the image loader in the background; allows John to get
		// his white background faster...
		Thread loader = new Thread(this);
		loader.start();

		setBackground(Color.white);
		setForeground(Color.black);
		setFont(new Font("Helvetica", Font.PLAIN, 12));
		setLayout(new BorderLayout());

		topBar = new Panel();
		topBar.setBackground(Color.white);
		topBar.add(newButton = new JButton(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.Button.NewRegion.Label")));
		topBar.add(selectButton = new JButton(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.Button.SetCorrect.Label")));
		topBar.add(deleteButton = new JButton(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.Button.Delete.Label")));
		newButton.addActionListener(this);
		selectButton.addActionListener(this);
		deleteButton.addActionListener(this);
		add("North", topBar);

		imageArea = new CICImageCanvas(getRegionsFromParameters(), getAnswerFromParameters(), this);
		add("Center", imageArea);
	}

	/**
	 * Handle the accelerator keys.
	 */
	public boolean keyDown(Event event, int key) {

		// JLO April 12 2000: Removed the requirement for ALT or META keys.
		// Allowed both lower and upper case

		switch (key) {

		case 'N':
		case 'n':
			imageArea.addRegion();
			return true;

		case 'D':
		case 'd':
			imageArea.deleteActiveRegion();
			return true;

		case 'S':
		case 's':
			imageArea.selectActiveRegion();
			return true;
		}
		return super.keyDown(event, key);
	}

	/**
	 * When the mouse comes in, show an appropriate message.
	 */
	public boolean mouseEnter(Event event, int x, int y) {

		this.showStatus(loadingStatus);
		return true;
	}

	private void popup(String message) {

		new applets.util.WarningDialog(this.holderFrame, message, mtaLocale).setVisible(true);
	}

	public void run() {

		try {
			image = this.getImage(completeURL(getParameter("imageURL")));
			MediaTracker tracker = new MediaTracker(this);
			tracker.addImage(image, 0);

			while (tracker.statusID(0, true) == MediaTracker.LOADING) {

				showStatus(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.LoadingStatus.LoadingImage"));
				try {
					Thread.sleep(500);
				} catch (InterruptedException e) {
				}
				showStatus("");
				try {
					Thread.sleep(500);
				} catch (InterruptedException e) {
				}
			}
			if (tracker.isErrorID(0))
				throw new Exception(applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.LoadingStatus.ErrorLoadingImage"));
			gotImage = true;
			showStatus(loadingStatus = applets.MessageUtilities.getMessage(mtaLocale, "ClickableImageCreator.LoadingStatus.ImageLoaded"));

			// fix transparent pixels by painting onto blank image...
			Image tempImage = this.createImage(image.getWidth(this), image.getHeight(this));
			tempImage.getGraphics().drawImage(image, 0, 0, Color.white, this);
			// image.flush(); netscape doesn't like this much ... (?)
			image = tempImage;

			if (imageArea != null)
				imageArea.repaint(); // get the imagearea to realize it has the image...
			// shouldn't call imageArea.gotImage since it may not yet be laid out, and won't know
			// its size
		} catch (Exception e) {
			dialog(e);
		}
	}

}
