package applets.grapher;

import java.util.Locale;
import java.util.Vector;

class Parser
    implements Cloneable, java.io.Serializable, Function
{
	protected Token op = new Token("0", number);
	protected Parser left = null;
	protected Parser right = null;
// FINAL VARIABLES

	/** Flag to allow multi-character variable names (one-character variables are the default) */
	final public short MULTI_CHAR_PER_VAR = 1;
/** Flag to use the restricted list of functions (the full list is the default). */
	final public int USE_RESTRICTED_FUNCTIONS = 2;
	final public int DEFAULT_MODE = 0;
	final String[][] CONSTANTS = { {"Pi", ""+Math.PI},
				       {"pi", ""+Math.PI},
				       {"e", ""+Math.E}};
	final private String[] FULL_FUNCTIONS = { "sin", "cos", "tan", 
						  "log", "ln", "exp",
						  "abs", "sqrt", 
						  "arcsin", "arccos", "arctan",
						  "asin", "acos", "atan",
						  "sec", "csc", "cot",
						  "fact", "erf","studentst", "binomial", "sum"};
	final private String[] RESTRICTED_FUNCTIONS = { "sqrt", "Sqrt" };
// The names of the different classes of token returned by tokenize()
	final protected static String number = "number";
	final protected static String variable = "variable";
	final protected static String operator = "operator";
	final protected static String unary = "unary";
	final protected static String function = "function";
	final protected static String openParen = "(";
	final protected static String closeParen = ")";
	final protected static String string = "string";

// The user's session locale. 
	private Locale mtaLocale = new Locale("en", "US");

// Constuctor that pickps up the session locale.
	public Parser(Locale theLocale) {
		mtaLocale = theLocale;
	}

	public Parser() {
	}
	
/**
Create a parser from the formula in the default modes
 */
public void parse(String f) {
    buildTree(tokenize(f, DEFAULT_MODE));
}


// make a new array which contains the elements of a from i to j, inclusive
private static Token[] arrayCut(Token[] a, int i, int j, Locale theLocale) {

	if( j-i+1 <= 0 ) {
		throw new NumberFormatException(applets.MessageUtilities.getMessage(theLocale, "Grapher.Parser.Error.ProblemWithExpression")+" " + tokenListToString(a));
	}
	Token[] r = new Token[j-i+1];

	for(int k=0; k<j-i+1; k++)
		r[k] = a[k+i];

	return r;
}

void buildTree(Token[] tokenList) {

	int breakPos = -1, parCount = 0, lastPrecedence = -1, currentPrecedence;


	// Make sure we erase any previous values (especially important since nodes are
	// created by cloning...

	left = null;
	right = null;


	if( tokenList==null || tokenList.length==0 )
		throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.FormulaHasNoSymbols"));

	// Run thru the token list, keeping track of the parenthesis level. When we find
	// an operator outside of all parentheses, note its position (overwriting the position
	// of any operators of the same type to the left of it)

	for(int i=0; i<tokenList.length; i++) {

		if( tokenList[i].type==openParen )
			parCount++;

		else if( tokenList[i].type==closeParen ) 
			parCount--;

		else if( parCount==0  && tokenList[i].type==operator )  {
			currentPrecedence = operatorPrecedence(tokenList[i]);

			if( currentPrecedence >= lastPrecedence ) {
				breakPos = i;
				lastPrecedence = currentPrecedence;
			}
		}
	}

	try {

		if( parCount > 0 ) {
			throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.NonMatchingParentheses")+" " +
			                                tokenListToString(tokenList) );
		}
		else if( parCount < 0 ) {
			throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.NotMatchingParentheses")+" " +
			                                tokenListToString(tokenList) );
		}

		if( tokenList[0].type==unary && lastPrecedence <= operatorPrecedence(tokenList[0]) ) {
			if( tokenList[0].text.equals("-") ) {
				this.op = tokenList[0];
				this.right = (Parser)super.clone();
				this.right.buildTree( arrayCut(tokenList, 1, tokenList.length-1, mtaLocale) );
			}
			else {
				this.buildTree( arrayCut(tokenList, 1, tokenList.length-1, mtaLocale) );
			}
		}
		else if( breakPos > 0 ) {
			this.left = (Parser)super.clone();
			this.left.buildTree(arrayCut(tokenList, 0, breakPos-1, mtaLocale));

			this.op = tokenList[breakPos];

			this.right = (Parser)super.clone();
			this.right.buildTree(arrayCut(tokenList, breakPos+1, tokenList.length-1, mtaLocale));
		}
		else if( tokenList.length==1 && (tokenList[0].type==variable || tokenList[0].type==number) ) {
			this.op = tokenList[0];
		}
		else if( tokenList[0].type==openParen && tokenList[tokenList.length-1].type==closeParen ) {
			this.buildTree(arrayCut(tokenList, 1, tokenList.length-2, mtaLocale));
		}
		else if( tokenList[0].type==function ) {
			this.op = tokenList[0];
			this.right = (Parser)super.clone();
			this.right.buildTree(arrayCut(tokenList, 1, tokenList.length-1, mtaLocale));
		}
		else if( tokenList[0].type == string ) {
		
			this.op = tokenList[0];
		
			if( tokenList.length > 1 ) {
			
				this.right = (Parser)super.clone();
				this.right.buildTree(arrayCut(tokenList, 1, tokenList.length-1, mtaLocale));
			}
		}
		else {
			throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.ProblemWithExpression")+" " + tokenListToString(tokenList));
		}
	}
	catch(CloneNotSupportedException e) {

		throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.ShouldNeverHappen"));
	}
}

/** 
 * Overrides the method in java.lang.Object.
 * Clones not just this Parser object but recurses down the tree to return
 * a copy of the full parser tree.
 * @exception  CloneNotSupportedException In fact this exception ought never to be thrown
 */
public Object clone() throws CloneNotSupportedException {

	Parser p = (Parser)super.clone();
	p.left = (p.left != null) ? (Parser)p.left.clone() : null;
	p.right = (p.right != null) ? (Parser)p.right.clone() : null;
	return p;
}

/** 
The basic evaluation routine. Since a completely evaluated tree yields a double, while
a partially evaluated one yields a string, this routine returns an Object, which may be cast to either
a Double or a String. Doing this prevents converting doubles to strings and back again.
 */
final public Object eval(String[] variables, double[] values) {
	Object leftValue, rightValue;

	if( this.op.type==variable ) {
		if( variables != null) 
			for(int i=0; i<variables.length; i++)
			if( op.text.equals(variables[i]) )
			return( new Double(values[i]) );

		return( this.op.text );
	}


	leftValue = (left==null ? null : left.eval(variables, values));
	rightValue = (right==null ? null : right.eval(variables, values));

	return( evaluate(leftValue, op, rightValue) );
}

// PUBLIC METHODS:
/**
Evaluate the formula as a function of the vaiable <tt>x</tt>. 
@return If the formula doesn't evaluates to a double, this returns Double.NaN
 */
final public double eval(double x) {
	String[] variable = {"x"};
	double[] value = {x};

	Object y = eval(variable, value);

	if( y instanceof Double )
		return(((Double)y).doubleValue());
	else
		return(Double.NaN);
}

/**
Performs the arithmetic evaluation of a single node. It is given the operation and the
values of the  left and right nodes, and it returns the appropriate Double or String. This 
is the method which subclasses should override to add new functionality to evaluation. NOTE that
evaluate() does not handle variable substitution. That is handled by eval(). In fact the names
and values of variables are not even available to evaluate().
@exception java.lang.IllegalArgumentException If it is passed a Token of type "variable"
 */
Object evaluate(Object leftValue, Token op, Object rightValue) {

	if( op.type==unary ) {

		if( rightValue instanceof Double ) 
			return new Double(-((Double)rightValue).doubleValue());

		else 
			return( "-(" + rightValue + ")" );
	}
	else if( op.type==operator ){

		if( leftValue instanceof Double && rightValue instanceof Double ) {
			double l = ((Double)leftValue).doubleValue();
			double r = ((Double)rightValue).doubleValue();

			switch( op.text.charAt(0) ) {
			case '+': 
				return( new Double( l + r ) );

			case '-': 
				return( new Double( l - r ) );

			case '*': 
				return( new Double( l * r ) );

			case '/': 
				return( new Double( l / r ) );

			case '^': 
				return( new Double( Math.pow(l, r) ) );
			}
		}
		else {

			String l = leftValue.toString();
			String r =  rightValue.toString();

			if( leftValue instanceof String )
				l = "(" + l + ")";

			if( rightValue instanceof String )
				r = "(" + r + ")";


			switch( op.text.charAt(0) ) {
			case '+': 
				return( l  + "+" + r );

			case '-': 
				return( l + "-" + r );

			case '*': 
				return(  l + "*" + r );

			case '/': 
				return( l + "/" + r );

			case '^': 
				return( l + "^" + r );
			}
		}
	}
	else if( op.type==function ) {

		if( rightValue instanceof Double ) {
			double x = ((Double)rightValue).doubleValue();

			try {
				if( op.text.equals("sin") )
					return( new Double( Math.sin(x) ) );

				else if( op.text.equals("cos" ) )
					return( new Double( Math.cos(x) ) );

				else if( op.text.equals("tan") )
					return( new Double( Math.tan(x) ) );

				else if( op.text.equals("log") )
					return( new Double( Math.log(x)/Math.log(10) ) );

				else if( op.text.equals("ln") )
					return( new Double( Math.log(x) ) );
				
				else if( op.text.equals("exp") ) 
					return new Double( Math.exp(x) );

				else if( op.text.equals("abs") )
					return( new Double( Math.abs(x) ) );

				else if( op.text.equals("sqrt") || op.text.equals("Sqrt") )
					return( new Double( Math.sqrt(x) ) );

				else if( op.text.equals("asin") || op.text.equals("arcsin") )
					return( new Double( Math.asin(x) ) );

				else if( op.text.equals("acos") || op.text.equals("arccos") )
					return( new Double( Math.acos(x) ) );

				else if( op.text.equals("atan") || op.text.equals("arctan") )
					return( new Double( Math.atan(x) ) );

				else if( op.text.equals("sec") )
					return new Double( 1/Math.cos(x) );

				else if( op.text.equals("csc") )
					return new Double( 1/Math.sin(x) );

				else if( op.text.equals("cot") )
					return new Double( 1/Math.tan(x) );

				else
					throw new ArithmeticException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.UnknownFunction")+" "+op.text);
			} 
			catch(ArithmeticException e) {
				if( e.toString().equals(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.UnknownFunction")+" "+op.text) )
					throw e;
				else {
					return( op.text + "(" + rightValue + ")" );
				}
			}
		}
		else {
			return( op.text + "(" + rightValue + ")" );
		}	
	}
	else if( op.type==number ) {
		return( new Double(op.text) );
	}
	else if( op.type==string ) {
		return( rightValue==null ? op.text : (op.text + rightValue));
	}
	else if( op.type==variable )
		throw new IllegalArgumentException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.InvalidToken"));

	else
		throw new ArithmeticException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.CannotEvaluateToken")+" " + op.type);

	return(null);
}

/**
Override this method to change the functions names that the parser recognizes. WARNING!! Do *not* override
this method unless you also override Parser.evaluate(), since the function string names that Parser recognizes
are hard-coded into the evaluation routine!!
 */
boolean foundFunctionIn(Token thisToken, boolean useFullFunctions) {

	String name = thisToken.text.toLowerCase();

	if( useFullFunctions ) {

		for(int i=0; i<FULL_FUNCTIONS.length; i++) {  

			if( name.equals(FULL_FUNCTIONS[i]) ) {

				thisToken.type = function;
				thisToken.text = name;
				return true;
			}
		}
	}
	else {

		// In restricted mode, first see if the token matches a name in the
		// restricted list. If it does, proceed. Otherwise, if it matches the name of a
		// full function, warn that this keyword cannot be used.
		for(int i=0; i<RESTRICTED_FUNCTIONS.length; i++) {
			if( name.equals(RESTRICTED_FUNCTIONS[i]) ) {

				thisToken.type = function;
				thisToken.text = name;
				return true;
			}
		}
		for(int i=0; i<FULL_FUNCTIONS.length; i++) {
			if( name.equals(FULL_FUNCTIONS[i]) ) {
				throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.IllegalUseOfFunction", thisToken.text));
			}
		}
	}
	return false;
}

private static boolean isAlpha(char c) { return( ( c>='a' && c<='z' ) || ( c>='A' && c<='Z' ) ); }

private static boolean isNum(char c) { return( c>='0' && c<='9' ); }

/**
verride this method to add new operators to subclasses. It should identify which characters can denote
operators.
 */
boolean isOperator(char c) { return( c=='+' || c=='-' || c=='*' || c=='/' || c=='^'); }

private static boolean isWhite(char c) { return( c==' ' || c=='\t' || c=='\n' || c=='\r' ); }

public static void main(String[] args) { 
    Parser p = new Parser();
    p.parse(args[0]);
    System.out.println(p); 
}

/**
Override this method to add new operators to subclasses. It should return a non-negative integer indicating
precedence of operators. Formulas will split first at higher precedence operators.
 */
int operatorPrecedence(Token t) {
	if( t.type == operator ) 
		switch( t.text.charAt(0) ) {
	case '^': return 1;
	case '/': return 3;
	case '*': return 4;
	case '-': return 5;
	case '+': return 6;
	default: throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.UnknownBinaryOperator", t));
	}
	else if( t.type == unary )
		return 2;

	else
		throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.BadBinaryOperator", t));
}

/** 
Override this method to use a different choice of constants (or none!)
 */
protected boolean substitutedForConstantsIn(Token thisToken) {

	for(int i=0; i<CONSTANTS.length; i++) {
		if( thisToken.text.equals(CONSTANTS[i][0]) ) {
			thisToken.text = CONSTANTS[i][1];
			thisToken.type = number;
			return true;
		}
	}
	return false;
}

/**
Presents the formula as a string. Note that the string will NOT be the same as was originally used
to construct the Parser; in particular, a lot of extra parenthesis will be present.
 */
public String toString() {

	return( eval(null, null).toString() );
}

// return a string obtained by pasting a token list together again
private static String tokenListToString(Token[] tokens) {
	String s = "";

	for(int i=0; i<tokens.length; s += tokens[i++].text);
	return s;
}

// This routine splits the given text string into an array of tokens. 
// The routine also does some parsing; constants are replaced with their numeric values, unary
// plus and minus are replaced with binary multiplications, and multiplications implied by juxtaposed
// symbols are made explicit.
private Token[] tokenize(String s, int mode) {
	Vector tokens = new Vector();
	int pos = 0;
	char c;


	// Interpret the conditions set in the mode...

	boolean useFullFunctions = true;

	if( (mode & USE_RESTRICTED_FUNCTIONS) == USE_RESTRICTED_FUNCTIONS) {
		useFullFunctions = false;
	}


	boolean oneCharPerVar = true;

	if( (mode & MULTI_CHAR_PER_VAR) == MULTI_CHAR_PER_VAR) {
		oneCharPerVar = false;
	}




	// Make sure that s ends in whitespace
	s += " "; 

	c = s.charAt(pos++); // Read in the first character and point to the next

	mainloop:
	while( pos < s.length() ) {
		if ( isWhite(c) ) {
			c = s.charAt(pos++);
		}
		else if( isAlpha(c) ) {
			// sb will hold the name of the alpha string and thisToken will hold the 
			// text of the token (possibly translated if it's a constant) and the type (contant.
			// function etc)
			StringBuffer sb = new StringBuffer();
			Token thisToken = new Token();

			// record the token
			tokens.addElement(thisToken);


			// NOTE: 9.18.97 did a major revision of this loop. Previous versions read the longest
			// alpha-num string starting at pos into sb, coverted it to a string in thisToken.text
			// and tested that maximal string to see if it was a function. Strings like "sqrt2",
			// "sinx" and (most seriously) "ex" would not be read correctly. Now we build up the
			// partial strings starting with one char and continuing till we meet a non-alpha char.
			// Each of the partials is tested against the list of functions and constants. If there's
			// a match, we translate the token appropriately. Otherwise, we treat the maximal string
			// as (starting with) a variable.

			do {
				sb.append(c);
				c = s.charAt(pos++);

				thisToken.text = sb.toString();
				
				// if it's the name of a function, record that fact

				if( foundFunctionIn(thisToken, useFullFunctions) ) { continue mainloop; }

			} while( isAlpha(c) /* || isNum(c)*/ );

			// JLO 12-11-00: The following two lines used to be part of the do-while loop above.
			// However in that case it substituted for constants (eg the letter "e") even if
			// they appeared as part of a function name (eg "eq()"). The present behavior
			// runs the risk that some constants will be missed, but I think it'll only happen
			// in very obscure situations, so is probably the best solution.
			//
			// if the token is the name of a constant, replace it with the value of that constant
			if( substitutedForConstantsIn(thisToken) ) { continue mainloop; }

			// Variables are not allowed in randomization
			// throw new NumberFormatException("The expression \"" + sb.toString() + "\" is not allowed.");
			
			// the only alternative is that it's the name of a variable
			thisToken.type = variable;

			// However, if we're only using one character variable names then
			// we must step back to the beginning of the chain of chars
			
			if( oneCharPerVar ) {
				pos -= thisToken.text.length();
				c = s.charAt(pos++);
				thisToken.text = "" + thisToken.text.charAt(0);	
			}
		}
		else if ( isNum(c) || c=='.' ) {
			StringBuffer sb = new StringBuffer();
			Token thisToken = new Token();

			// record the token
			tokens.addElement(thisToken);

			// if c was a number, read in all the subsequent numbers
			if( isNum(c) )
				do {
				sb.append(c);
				c = s.charAt(pos++);
			} while( isNum(c) );

			// if we get to a '.' then add it in and read in all the numbers following it
			if( c == '.' )
				do {
				sb.append(c);
				c = s.charAt(pos++);
			} while( isNum(c) );

			// if we get an exponent, record the 'E'...
			if( c == 'E' ) {
				sb.append('E');
				c = s.charAt(pos++);

				//TWF 2-8-99 modification to skip whitespace
				for(; isWhite(c) && (pos < s.length()); c = s.charAt(pos++) );
				
				// ...exponents are allowed to begin with a '-'...
				if( c=='-' ) {
					sb.append(c);
					c = s.charAt(pos++);
				}

				//TWF 2-225-99: allow exponents to begin with a '+' as well...
				if( c=='+' ) c=s.charAt(pos++);
				
				// ...but an exponent must have some digits, otherwise it's an error
				if( isNum(c) )
					do {
					sb.append(c);
					c = s.charAt(pos++);
				} while( isNum(c) );
				else
					throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.BadNumberFormat", sb));	
			}

			thisToken.text = sb.toString();
			thisToken.type = number;
		}
		else {

			Token thisToken = new Token(""+ c, "");

			if ( c=='(' ) 
				thisToken.type = openParen;

			else if ( c==')' )
				thisToken.type = closeParen;

			else if ( isOperator(c) ) 
				thisToken.type = operator;
			
			else if( c=='\"' ) {
			
				StringBuffer sb = new StringBuffer();
				
				for( ; pos < s.length()  &&  (c = s.charAt(pos)) != '\"'; pos++) sb.append(c);

				thisToken.type=string;
				thisToken.text = sb.toString();
				
				pos++;
			}
			else
				throw new NumberFormatException(applets.MessageUtilities.getMessage(mtaLocale, "Grapher.Parser.Error.IllegalCharacterInFormula", c));

			tokens.addElement(thisToken);
			c = s.charAt(pos++);
		}
	}

	// now we replace all unary +/- by pairs of tokens; (+1)(*) or (-1)(*)
	// and replace any juxtaposed symbols which imply multiplication with
	// explicit multiplication

	Token m = new Token("*", operator); // here's the multiplication token that we'll paste in

	for(int i=0; i<tokens.size(); i++) {

		// get the i-th token and its predecessor (assuming it has one)
		Token a = (i==0)?null:(Token)(tokens.elementAt(i-1));
		Token b = (Token)(tokens.elementAt(i));

		// locate unary +/-
		if( (i==0 || a.type==openParen || a.type==operator || a.type==unary) && (b.text.equals("+") || b.text.equals("-")) && b.type != string) {
			b.type = unary;
		}
		// locate implied multiplications
		else if( i>0 && (a.type==variable || a.type==number || a.type==closeParen) && !(b.type==operator || b.type==closeParen) )
			tokens.insertElementAt(m, i++);
	}
	
	// JLO Aug 30 2000
	// Merge unary-number sequences into a single signed number
	{
		Vector v = new Vector();
		for (int i=0; i<tokens.size(); i++) {
			Token a = (Token) tokens.elementAt(i);
			
			if (a.type == unary && i + 1 < tokens.size()) {
				Token b = (Token) tokens.elementAt(i + 1);
				
				if( b.type == number ) {
					b.text = a.text + b.text;
					v.addElement(b);
					i++;
					continue;
				}
			}
			v.addElement(a);
		}
		tokens = v;
	}

	Token r[] = new Token[tokens.size()];
	tokens.copyInto(r);
	return(r);
}
}


class Token
{
	String type;
	String text;

Token () {

}

Token (String text, String type) {

	this.type = type;
	this.text = text;
}

public String toString() {

	return( type + "  " + text );
}
}
