package applets.grapher;

import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.image.*;
import java.text.DecimalFormat;
import java.util.*;


/**
 * A GraphWindow is a Canvas for drawing the graphs of mathematical functions. A set of
 * axes is described by the ranges <tt>xMin, xMax</tt> and <tt>yMin, yMax</tt> (with tickmarks
 * spaced at intervals of <tt>xScale</tt> and <tt>yScale</tt> respectively on the x and y axes).
 * Functions are added to the window with the <tt>addFunction()</tt> method and removed by <tt>deleteFunction</tt>
*/
public  class GraphWindow extends Canvas {

	/*
	  <pre>	xMin, xMax, yMin, yMax specify the graphing area
	  xScale and yScale specify where the tickmarks go. No marks on axis if zero.
	  boolean	xAxisLabel and yAxisLabel say whether or not to put numeric labels
	  against tickmarks
	  Color	graphColor is the color the function is drawn in
	  boolean	showGrid says whether to show a grid based on the tickmarks
	  Color	gridColor says what color this grid should be
	  </pre>
	*/

	/** Margin around the canvas */
	private static final int MARGIN = 25;

	/** The font to label the axes in */
	private Font axesFont = new Font("Arial", Font.PLAIN, 8);
	
	/** The number format for the axis labels. */
	private DecimalFormat decfmt = new DecimalFormat("#,###.##");
	private DecimalFormat expfmt = new DecimalFormat("0.##E0");
	private double maxDecFmt = 9999.0;
	private double minDecFmt = 0.05;
	/** User-specified numer format for the x-axis labels. */
	private DecimalFormat xfmt;
	/** User-specified numer format for the y-axis labels. */
	private DecimalFormat yfmt;

	/** The font for labels to the graph. */
	private Font labelFont = new Font("Arial", Font.PLAIN, 12);

	/** Left-hand endpoint of the graph */
	protected double xMin;

	/** Right-hand endpoint of the graph */
	protected double xMax;

	/** Spacing of the tickmarks on the x-axis */
	protected double xScale;

	/** Bottom endpoint of the graph */
	protected double yMin;

	/** Top endpoint of the graph */
	protected double yMax;

	/** Spacing of the tickmarks on the y-axis */
	protected double yScale;

	/** Color of the optional grid */
	protected Color gridColor = new Color(0xCC, 0xCC, 0xFF);

	/** Color of the optional grid */
	protected Color axisColor = new Color(0x50, 0x50, 0x50);

	/** Label the x-axis with numbers if set */
	protected boolean xAxisLabel;

	/** Label the y-axis with numbers if set */
	protected boolean yAxisLabel;

	/** Show a background grid if set */
	protected boolean showGrid;
	
	/** List of all the functions on the graph. */
	protected Vector functionList = new Vector();
	private Hashtable functionColors = new Hashtable();
    
	private Hashtable labelList = new Hashtable();
	
	/** The default list of colors that functions are  drawn in */
	private static Hashtable colorList = new Hashtable();

	/** Constant value to allow any number of functions. */
	public static final int UNLIMITED = 0;
	
	/** The number of functions allowed in this window.*/
	public int maxFunctions = UNLIMITED;

	private Image background;
	private Image offscreen;
    
	/** Used for dragging operations. */
	private int lastx;
	/** Used for dragging operations. */
	private int lasty;

	/**
	   Constructs a GraphWindow with the specified ranges of x and y axes. The axes show no tickmarks.
	*/
	public GraphWindow(double xMin, double xMax, double yMin, double yMax){
	this(xMin,xMax,0.0,yMin,yMax,0.0);
	}

	/**
	   Constructs a GraphWindow with the specifies ranges of x and y axes. 
	   The axes show tickmarks marked out at intervals of <tt>xScale</tt> and <tt>yScale</tt>.
	*/
	public GraphWindow(double xMin, double xMax, double xScale, double yMin, double yMax, double yScale){
	super();
	setBackground(Color.white);
	this.xMax=xMax;
	this.xMin=xMin;
	this.yMax=yMax;
	this.yMin=yMin;
	this.xScale=xScale;
	this.yScale=yScale;
	this.xAxisLabel = true;
	this.yAxisLabel = true;
	this.showGrid = true;
		
	// Create the default graphing colors and arrange them in a cycle, so that each color points
	// to the next one round.
	Color c0 = new Color(255, 0, 0);
	Color c1 = new Color(0, 150, 0);
	Color c2 = new Color(0, 0, 220);
	Color c3 = new Color(200, 0, 200);
	Color c4 = new Color(212,212,0);
	Color c5 = new Color(0,125,125);
	colorList.put(c0,c1);
	colorList.put(c1,c2);
	colorList.put(c2,c3);
	colorList.put(c3,c4);
	colorList.put(c4,c5);
	colorList.put(c5,c0);
	
	addMouseListener(new MouseAdapter() {
		public void mousePressed(MouseEvent e) {
			lastx = e.getX();
			lasty = e.getY();
			update(getGraphics());
		}
	});
	addMouseMotionListener(new MouseMotionAdapter() {
		public void mouseDragged(MouseEvent e) {
			int mods = e.getModifiers();
			int x = e.getX();
			int y = e.getY();
			if ((mods & MouseEvent.SHIFT_MASK) == MouseEvent.SHIFT_MASK) {
				doShift(x, y);
			} else if ((mods & MouseEvent.CTRL_MASK) == MouseEvent.CTRL_MASK) {
				doDilate(x, y);
			} else {
				doTrace(x, y);
			}
			lastx = x;
			lasty = y;
		}
	});
	}

	/**
	 * Make a GraphWindow with the default graphing range of -5...5 and -5...5
	 */
	public GraphWindow() {this(-5.0, 5.0, 1.0, -5.0, 5.0, 1.0);}

	public void setBackground(Image im) {
	background = im;
	repaint();
	}
    
	public void setXFmt(String fmt) {
		xfmt = new DecimalFormat(fmt);
	}
    
	public void setYFmt(String fmt) {
		yfmt = new DecimalFormat(fmt);
	}

	/**
	 * Change the axes and scales and redraw the graph from scratch.
	 */
	public void drawGraph(double xMin, double xMax, double xScale, double yMin, double yMax, double yScale){
	this.xMax=xMax;
	this.xMin=xMin;
	this.yMax=yMax;
	this.yMin=yMin;
	this.xScale=xScale;
	this.yScale=yScale;
	repaint();
	}

	/**
	   Change the axes, set the scale to zero (ie no tickmarks), and redraw from scratch.
	*/
	public void drawGraph(double xMin, double xMax, double yMin, double yMax){
	drawGraph(xMin,xMax,0.0,yMin,yMax,0.0);
	}
	
	/**
	   Add a new Function to the list of functions being drawn, and draw it in the given Color.
	*/
	public void addFunction(Function f, Color c) {
	if( this.maxFunctions != this.UNLIMITED  &&  functionList.size() >= this.maxFunctions )
		deleteFunction( (Function)functionList.elementAt(0) );    // This lets subclasses do their tidying too....

	functionColors.put(f, c);
	functionList.addElement(f);
	}

	/**
	   Add a new Function to the list of functions being drawn, and draw it in the appropriate default color.
	*/
	public void addFunction(Function f) {
	if( functionList.size() == 0 )
		addFunction(f, (Color)colorList.keys().nextElement());
	else
		addFunction(f, (Color)colorList.get(functionColors.get( functionList.lastElement() )));
	}
		
	
	/**
	   Remove the function from the list of functions being drawn.
	*/
	public void deleteFunction(Function f) {
	functionList.removeElement(f);
	functionColors.remove(f);
	}
	
	
	/**
	   Delate the last function to have been added from the list of functions being drawn.
	*/
	public void deleteLastFunction() {
	deleteFunction((Function)(functionList.lastElement()));
	}
    
	public void addLabel(double x, double y, String text) {
		labelList.put(new DoublePoint(x, y), text);
	}
	
	/**
	   Empty the list of functions being drawn.
	*/
	public void deleteAll() {
	functionColors = new Hashtable();
	functionList = new Vector();
	repaint();
	}
	
	/**
	   Returns the value of xMin
	*/
	public double xMin() {
	return xMin;
	}


	/**
	   Returns the value of xMax
	*/
	public double xMax() {
	return xMax;
	}


	/**
	   Returns the value of xScale
	*/
	public double xScale() {
	return xScale;
	}


	/**
	   Returns the value of yMin
	*/
	public double yMin() {
	return yMin;
	}


	/**
	   Returns the value of yMax
	*/
	public double yMax() {
	return yMax;
	}


	/**
	   Returns the value of yScale
	*/
	public double yScale() {
	return yScale;
	}



	/**
	   Convert x-values to pixels. 
	   Convert a double (presumed in the range xMin to xMax) to the corresponding x coordinate
	   in the current graphics context. 
	*/
	public int x(double xValue) {
		return (int) (MARGIN + (size().width - 2*MARGIN) * (xValue - xMin)/(xMax - xMin) + 0.5);
	}
	
	/**
	   Convert pixels to an x-value. 
	   Convert an integer x coordinate (in the current graphics context) to the corresponding double 
	   in the range xMin to xMax. 
	*/
	public double x(int xCoord) {
		return (xMax - xMin) * ((double) xCoord - MARGIN)/(size().width - 2*MARGIN) + xMin;
	}
	
	/**
	   Convert y-values to pixels.
	   Convert a double (presumed in the range yMin to yMax) to the corresponding y coordinate
	   in the current graphics context. 
	*/
	public int y(double yValue) {
		return (int) (MARGIN + (size().height - 2*MARGIN) * (yMax - yValue)/(yMax - yMin) + 0.5);
	}
	
	/**
	   Convert pixels to y-values.
	   Convert an integer y coordinate (in the current graphics context) to the corresponding double 
	   in the range yMin to yMax. 
	*/
	public double y(int yCoord) {
		return yMax - (yMax - yMin) * ((double) yCoord - MARGIN)/(size().height - 2*MARGIN);
	}
	
	public void update(Graphics g) {
		paint(g);
	}
	
	public void paint(Graphics g) {
		if (offscreen == null) {
			offscreen = createImage(size().width, size().height);
			doPaint(offscreen.getGraphics());
		}
		g.drawImage(offscreen, 0, 0, size().width, size().width, this);
	}
    
	private void doPaint(Graphics g) {
	g.setColor(Color.white);
	g.fillRect(0,0,this.size().width,this.size().height);
	if (background != null) {
		g.drawImage(background, 0, 0, this);
	}
	if (showGrid) {
		drawGrid(g);
	}
	drawAllFunctions(g);
	drawAxes(g);
	drawLabels(g);		
	}
	
	
	private void drawAllFunctions(Graphics g) {
	Function f;
		
	for (Enumeration e = functionColors.keys(); e.hasMoreElements(); ) {
		f = (Function) e.nextElement();
		drawFunction(g, f, (Color) functionColors.get(f)) ;
	}
	}
    
	private void drawLabels(Graphics g) {
		g.setFont(labelFont);
		FontMetrics fm = g.getFontMetrics();
    	
		for (Enumeration e = labelList.keys(); e.hasMoreElements(); ) {
			DoublePoint p = (DoublePoint) e.nextElement();
			String label = (String) labelList.get(p);
			drawLabel(label, g, x(p.x), y(p.y));	
		}
	}
    
	private void drawLabel(String label, Graphics g,int x, int y) {
		FontMetrics fm = g.getFontMetrics();
		int width = fm.stringWidth(label);
		int ascent = fm.getAscent();
		int descent = fm.getDescent();
		g.setColor(getBackground());
		g.fillRect(x, y - ascent, width, ascent + descent);
		g.setColor(getForeground());
		g.drawString(label, x, y);
	}

	private void drawFunction(Graphics g, Function f, Color graphColor) {
		if (xMax <= xMin || yMax <= yMin) {
			return;
		}
		
		int width = size().width - 2 * MARGIN;
		int height = size().height - 2 * MARGIN;
		int lastPoint = 0;
		boolean lastPointExists = true;

		g.setColor(graphColor);

		try {
			lastPoint = y(f.eval(xMin));
		} catch (Exception e) {
			lastPointExists = false;
		}

		for (int i = MARGIN; i < width + MARGIN; i++) {
			try {
				double x = x(i);
				double y = f.eval(x);
				int j = y(y);

				if (j < MARGIN || j > height + MARGIN) {
					lastPointExists = false;
					continue;
				} else if (lastPointExists) {
					g.drawLine(i - 1, lastPoint, i, j);
				} else {
					lastPointExists = true;
				}
				lastPoint = j;
			} catch (Exception e) {
				lastPointExists = false;
			}
		}
	}



	private void drawAxes(Graphics g){
	int width = this.size().width - 2*MARGIN;
	int height = this.size().height - 2*MARGIN;
	FontMetrics theFontMetrics = getFontMetrics(axesFont);

	g.setFont(axesFont);
	g.setColor(axisColor);
	
	int x0;
	if (xMin > 0.0) {
		x0 = 0;
	} else if (xMax < 0.0) {
		x0 = x(xMax) + 4;
	} else {
		x0 = x(0.0);
	}
	int y0;
	if (yMin > 0.0) {
		y0 = y(yMin) + 4;
	} else if (yMax < 0.0) {
		y0 = 0;
	} else {
		y0 = y(0.0);
	}

	g.drawLine(x0, y(yMin), x0, y(yMax));
	if (xAxisLabel) {
		int lo = (int) Math.ceil(xMin/xScale);
		int hi = (int) Math.floor(xMax/xScale);
		for (int k = lo; k <= hi; k++) {
			double x = xScale * k;
			int i = x(x);
			String label = formatX(x);
			int w = theFontMetrics.stringWidth(label);
			if (w < x(xScale) - x(0.0) - 2) {
				drawLabel(label, g, i - w/2, y0 + 10);
			}
			g.drawLine(i, y0, i, y0 + 2);
		}
	}

	g.drawLine(x(xMin), y0, x(xMax), y0);
	if (yAxisLabel) {
		int lo = (int) Math.ceil(yMin/yScale);
		int hi = (int) Math.floor(yMax/yScale);
		for (int k = lo; k <= hi; k++) {
			if (k == 0) {
				continue;
			}
			double y = yScale * k;
			int j = y(y);
			String label = formatY(y);
			int h = theFontMetrics.getAscent();
			System.out.println(h + " > " + (y(0.0) - y(yScale)) + " (" + yScale + ")");
			if (h < y(0.0) - y(yScale)) {
				drawLabel(label, g, x0 + 6, j + h/2);
			}
			g.drawLine(x0, j, x0 + 2, j);
		}
	}
	}


	private void drawGrid(Graphics g) {
		g.setColor(gridColor);
		
		int min = y(yMin);
		int max = y(yMax);
		int lo = (int) Math.ceil(xMin/xScale);
		int hi = (int) Math.floor(xMax/xScale);
		for (int k = lo; k <= hi; k++) {
			int i = x(xScale * k);
			g.drawLine(i, min, i, max);
		}

		min = x(xMin);
		max = x(xMax);
		lo = (int) Math.ceil(yMin/yScale);
		hi = (int) Math.floor(yMax/yScale);
		for (int k = lo; k <= hi; k++) {
			int j = y(yScale * k);
			g.drawLine(min, j, max, j);
		}
	}
	
	private void doTrace(int x, int y) {
		x -= 4;
		y -= 4;
		
		String label = "(" + formatX(x(x)) + ", " + formatY(y(y)) + ")";
		Graphics g = getGraphics();
		update(g);
		g.setFont(labelFont);
		drawLabel(label, g, x + 10 , y + 3);
		g.drawLine(x-5, y, x-2, y);
		g.drawLine(x+2, y, x+5, y);
		g.drawLine(x, y-5, x, y-2);
		g.drawLine(x, y+2, x, y+5);
	}
	
	private void doShift(int x, int y) {
		double dx = x(x) - x(lastx);
		double dy = y(y) - y(lasty);
		xMin -= dx;
		xMax -= dx;
		yMin -= dy;
		yMax -= dy;
		offscreen = null;
		update(getGraphics());
	}
	
	private void doDilate(int x, int y) {
		int w2 = size().width/2;
		int h2 = size().height/2;
		int i = lastx - w2;
		int j = lasty - h2;
		double r1 = Math.sqrt(i*i + j*j);
		i = x - w2;
		j = y - h2;
		double r2 = Math.sqrt(i*i + j*j);
		double q = r2/r1;
		double xMid = (xMin+ xMax)/2.0;
		double yMid = (yMin + yMax)/2.0;
		xMin = xMid + q*(xMin - xMid);
		xMax = xMid + q*(xMax - xMid);
		yMin = yMid + q*(yMin - yMid);
		yMax = yMid + q*(yMax - yMid);
		offscreen = null;
		update(getGraphics());
	}
	
	private String formatX(double x) {
		if (xfmt != null) {
			return xfmt.format(x);
		} else {
			return format(x);
		}
	}
	
	private String formatY(double y) {
		if (yfmt != null) {
			return yfmt.format(y);
		} else {
			return format(y);
		}
	}

	private String format(double x) {
		double a = Math.abs(x);
		if (a == 0.0) {
			return "0";
		} else if (a < minDecFmt) {
			return expfmt.format(x);
		} else if (a > maxDecFmt) {
			return expfmt.format(x);
		} else {
			return decfmt.format(x);
		}
	} 
}
		
