
#include "com_maplesoft_client_MapleSocket.h"

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <stdio.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/param.h>

#if defined APPLE_PPC_OSX || defined IBM_RISC_UNIX
#include <netinet/in.h>
#endif

#define CHECK_EXCEPTION( jenv, ret ) 	if ( (*(jenv))->ExceptionOccurred( (jenv) ) ) { return (ret); }
#define CHECK_EXCEPTION_N( jenv ) 	if ( (*(jenv))->ExceptionOccurred( (jenv) ) ) { return; }

#if defined _DEBUG_
#define DBG( cmd )			(cmd)
#else
#define DBG( cmd )			
#endif

static jfieldID id;
static int init = 1;

static int getSocket( JNIEnv *jenv, jobject obj )
{
    int socket;

    if ( init )
    {
	jclass cls;

	cls = (*jenv)->FindClass( jenv, 
		"com/maplesoft/client/MapleSocket" );
	CHECK_EXCEPTION( jenv, -1 );
	
	id = (*jenv)->GetFieldID( jenv, cls, "socket", "I" );
	CHECK_EXCEPTION( jenv, -1 );

	init = 0;
    }

    socket = (int)((*jenv)->GetIntField( jenv, obj, id ));
    CHECK_EXCEPTION( jenv, -1 );

    return socket;
}

static void setSocket( JNIEnv *jenv, jobject obj, int value )
{
    if ( init )
    {
	jclass cls;

	cls = (*jenv)->FindClass( jenv, 
		"com/maplesoft/client/MapleSocket" );
	CHECK_EXCEPTION_N( jenv );
	
	id = (*jenv)->GetFieldID( jenv, cls, "socket", "I" );
	CHECK_EXCEPTION_N( jenv );

	init = 0;
    }

    (*jenv)->SetIntField( jenv, obj, id, value );
    CHECK_EXCEPTION_N( jenv );

    return;
}

static void throwIOException( JNIEnv *jenv, const char *c_msg )
{
    jclass cls;

    cls = (*jenv)->FindClass( jenv, "java/io/IOException" );
    CHECK_EXCEPTION_N( jenv );

    (*jenv)->ThrowNew( jenv, cls, c_msg );
}

static void throwIOExceptionErrno( JNIEnv *jenv, const char *msg )
{
    int len;
    char *buf;

    len = strlen( msg ) + strlen( strerror( errno ) )+3;

    buf = (char*)malloc( len );

    sprintf( buf, "%s: %s", msg, strerror( errno ) );
    
    throwIOException( jenv, buf );

    free( buf );
}

/*
 * Class:     MapleSocket
 * Method:    nativeConnect
 * Signature: (Ljava/lang/String;I)V
 */
JNIEXPORT void JNICALL Java_com_maplesoft_client_MapleSocket_nativeConnect(JNIEnv *jenv, jobject obj, 
	jstring address, jint port)
{
    struct sockaddr_in connector;
    struct hostent *hp;
    const char *addr;
    int sock;
#if 0
    int flags;
#endif

    sock = socket( AF_INET, SOCK_STREAM, 0 );

    if ( sock == -1 )
    {
	throwIOExceptionErrno( jenv, "could not create socket" );

	return;
    }

    /*connector = (struct sockaddr_in *)malloc( sizeof(struct sockaddr_in) );*/

    connector.sin_family = AF_INET;

    addr = (*jenv)->GetStringUTFChars( jenv, address, NULL );
    CHECK_EXCEPTION_N( jenv );

    hp = gethostbyname( addr );

    (*jenv)->ReleaseStringUTFChars( jenv, address, addr );
    CHECK_EXCEPTION_N( jenv );
    addr = NULL;

    if ( hp == NULL )
    {
	throwIOExceptionErrno( jenv, "unable to get address of host" );
	/*free( connector );*/
	return;
    }

    memcpy( (void*)(&connector.sin_addr), (void*)hp->h_addr, hp->h_length );
    connector.sin_port = htons( port );

    if ( connect( sock, (struct sockaddr *)&connector, sizeof(connector) ) < 0 )
    {
	printf( "1\n" );
	throwIOExceptionErrno( jenv, "could not connect socket to port" );
	/*free( connector );*/
	return;
    }

#ifdef TCP_NODELAY
    {
	/* disable buffering */
	/* Disables the Nagle algorithm for send coalescing */
	int flag = 1;

	if( setsockopt(sock,IPPROTO_TCP,TCP_NODELAY, 
				    (char*)&flag,sizeof(int)) < 0 ) 
	{
	    DBG( perror("setting TCP_NODELAY"); )
	}
    }
#endif

#ifdef SO_LINGER
    {
	struct linger ls;
	ls.l_onoff = 1;
	ls.l_linger = 2; /* 2 seconds */

	if( setsockopt(sock,SOL_SOCKET,SO_LINGER,(char*)&ls,sizeof(ls)) < 0 ) 
	{
	    DBG( perror("setting SO_LINGER") );
	}
    }
#endif

#ifdef SOCKET_BUFFER_SIZE
    {
	/* set the socket packet size */
	int flag = SOCKET_BUFFER_SIZE;
	if( setsockopt(sock,SOL_SOCKET,SO_RCVBUF,(char*)&flag,sizeof(int))<0 )
	{
	    DBG( perror("setting SOCKET_BUFFER_SIZE") );
	}
    }
#endif

#if 0 /* use blocking */

    /* If they have O_NONBLOCK, use the Posix way to do it */
#if defined(O_NONBLOCK)
    /* Fixme: O_NONBLOCK is defined but broken on SunOS 4.1.x and AIX 3.2.5. */
    if( -1 == (flags=fcntl(sock,F_GETFL,0)) )
    {
	flags = 0;
    }

    if ( fcntl(sock,F_SETFL,flags|O_NONBLOCK) == -1 )
    {
	throwIOExceptionErrno( jenv, "could not set socket non-blocking" );
	/*free( connector );*/
	return;
    }
#else
    /* Otherwise, use the old way of doing it */
    flags = 1;
    if ( ioctl(sock,FIOBIO,&flags) == -1 )
    {
	throwIOExceptionErrno( jenv, "could not set socket non-blocking" );
	/*free( connector );*/
	return;
    }
#endif

#endif

    setSocket( jenv, obj, sock );
    CHECK_EXCEPTION_N( jenv );

    return;
}

/*
 * Class:     MapleSocket
 * Method:    nativeDisconnect
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_com_maplesoft_client_MapleSocket_nativeDisconnect(JNIEnv *jenv, jobject obj)
{
    int socket;

    socket = getSocket( jenv, obj );
    CHECK_EXCEPTION_N( jenv );

    if ( socket == -1 )
    {
	throwIOException( jenv, "socket is not connected" );
	return;
    }

    if ( close( socket ) == -1 )
    {
	throwIOExceptionErrno( jenv, "error closing socket" );
	return;
    }

    setSocket( jenv, obj, -1 );
    CHECK_EXCEPTION_N( jenv );

    return;
}

/*
 * Class:     MapleSocket
 * Method:    nativeWrite
 * Signature: (I)V
 *
 * write the byte to the socket
 */
JNIEXPORT void JNICALL Java_com_maplesoft_client_MapleSocket_nativeWrite(
	JNIEnv *jenv, jobject obj, jint d)
{
    int socket;
    int ret;
    jbyte data;

    socket = getSocket( jenv, obj );
    CHECK_EXCEPTION_N( jenv );

    if ( socket == -1 )
    {
	throwIOException( jenv, "socket not connected" );
	return;
    }

    data = (jbyte)d;
    ret = write( socket, &data, 1 );

    if ( ret == -1 )
    {
#if 0
	if ( errno == EAGAIN )
	{
	    /* ? */
	}
#endif

	throwIOExceptionErrno( jenv, "error writing to socket" );
	return;
    }
    else if ( ret == 0 )
    {
	throwIOException( jenv, "nothing written to socket" );
	return;
    }
}

/*
 * Class:     MapleSocket
 * Method:    nativeRead
 * Signature: ()I
 *
 * return the byte as an integer value in the range 0-255, return -1 on 
 * eof
 */
JNIEXPORT jint JNICALL Java_com_maplesoft_client_MapleSocket_nativeRead(JNIEnv *jenv, jobject obj)
{
    int socket;
    int ret;
    unsigned char buffer;

    socket = getSocket( jenv, obj );
    CHECK_EXCEPTION( jenv, -1 );

    /*printf( "read from (%d)\n", socket );
    fflush( stdout );*/

    if ( socket == -1 )
    {
	throwIOException( jenv, "socket not connected" );
	return -1;
    }

    ret = recv( socket, (void*)(&buffer), 1, 0 );

    if ( ret == -1 )
    {
#if 0
	if ( errno == EAGAIN )
	{
	    /* ? */
	}
#endif

	throwIOExceptionErrno( jenv, "error reading from socket" );
	return -1;
    }

    if ( ret == 0 )
    {
	/* should never happen
	throwIOException( jenv, "no bytes read from socket" );*/
	return -1;
    }

    return (jint)(buffer);
}

/*
 * Class:     MapleSocket
 * Method:    nativeSendOOBData
 * Signature: (I)V
 *
 * although d is an int it stores a byte sized value
 */
JNIEXPORT void JNICALL Java_com_maplesoft_client_MapleSocket_nativeSendOOBData(JNIEnv * jenv, 
		jobject obj, jint d)
{
    int socket;
    int ret;
    unsigned char data;

    socket = getSocket( jenv, obj );
    CHECK_EXCEPTION_N( jenv );

    if ( socket == -1 )
    {
	throwIOException( jenv, "socket not connected" );
	return;
    }

    data = (unsigned char)(d & 0xff);

    ret = send( socket, (void*)(&data), 1, MSG_OOB );

    if ( ret == -1 )
    {
#if 0
	if ( errno == EAGAIN )
	{
	    /* ? */
	}
#endif

	throwIOExceptionErrno( jenv, "error sending OOB message" );
    }
    else if ( ret == 0 )
    {
	/* should never happen */
	throwIOException( jenv, "no byes send in OOB message" );
    }

    return;
}
